/*******************************************************************
*  MINARET (for edge-triggered circuits)
*
*  BY
*
*  NARESH MAHESHWARI AND SACHIN S. SAPATNEKAR
*
*  Copyright 1998 Iowa State University Research Foundation, Inc.
*  All Rights Reserved
*
*  Source Code for retiming edge-triggered circuits
*
*  ASTRA:   Refer to paper in DAC 95 by Deokar & Sapatnekar
*                       and IEEE Tran on CAD 10/1998
*  Minaret: Refer to paper in DAC 97 by Maheshwari & Sapatnekar
*                       and IEEE Tran on VLSI 10/1998
*
*  Contact Address: sachin@ece.umn.edu
*
*        Availiable on as is basis, with no support
*******************************************************************/
#include "include.h"
/*******************************
FILE util.c
misc utility functions 
******************************/


#ifdef SAFE

/*******************************
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
functions for memory allocation 
Used only in SAFE mode, otherwise macros in "include.h" 
are used, which unlike these functions do not check for
running out of memory
******************************/

void *CALLOC(size_t nelem, size_t elsize)
{
  void *t;
  t = calloc(nelem,elsize);
  if(t== NULL){printf(" ERR OUT OF MEM \n");exit(-1);}
  return (t);
}

void *MALLOC(size_t nelem, size_t elsize)
{
  void *t;
#ifdef MAL2CAL /** calloc initilizes memory, so is safer than malloc **/
  t = calloc(nelem,elsize);
#else
  t = malloc(nelem * elsize);
#endif
  if(t== NULL){printf(" ERR OUT OF MEM \n");exit(-1);}
  return (t);
}
/*******************************
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
******************************/

#endif

/****************************
Initilization of some global variables to be done
even before parsing the input netlist
*************************/
void init_ckt_stuff()
{
  nogates=0;	/* # gates */
  nolatches=0;	/* # Latches */
  nopin=0;	/* # Primary Inputs */
  nopout=0;	/* # Primary Outputs */
  noart=0;	/* # artifical gates*/
  nonodes=0;    /* # nodes */
  change = 0;
  nofreenodes = 0;

  OBJ_VALUE =0;

  /*** this array is just for debugging; so that the
    gate type can be printed as a string rather than
    as a integer ***/
  strcpy(Node_Type[0],"INVALID");
  strcpy(Node_Type[1],"PIN");
  strcpy(Node_Type[2],"POUT");
  strcpy(Node_Type[3],"AND");
  strcpy(Node_Type[4],"OR");
  strcpy(Node_Type[5],"NOT");
  strcpy(Node_Type[6],"NOR");
  strcpy(Node_Type[7],"NAND");
  strcpy(Node_Type[8],"DFF");
  strcpy(Node_Type[9],"HOST");
  strcpy(Node_Type[10],"ARTIF");

}

/*#####################################
QUEUE ROUTINES
#######################################*/

/*********************************************
 createq

initialize the queue ("top" and "bot" are global) with "k"
*********************************************/
void createq(int k)
{
  top = (Q *)MALLOC(1,sizeof(Q));
  bot = top;
  top->el = k;
  top->next = NULL;
}

/*********************************************
 addq

add "k" to the queue 
*********************************************/
void addq(int k)
{
  Q *q1;
  if(bot == NULL){
    createq(k);
    return;
  }

  q1 = ( Q *)MALLOC(1,sizeof(Q));
  q1->el = k;
  bot->next = q1;
  q1->next = NULL;
  bot = q1;
}

/*********************************************
 removeq

remove the top element from the queue
*********************************************/
void removeq()
{
  Q *q1;
  q1 = top;
  top = top->next;
  if(top == NULL)bot = NULL;
  free(q1);
}


/*******************************
NOT in use, see delete_w

deletes all occurances of "n" from the list 
VERY VERY time consuming, since it scans the 
whole list every time.
******************************/

W * Sdelete_W(W *fout,int n)
{
  W *w1, *prev, *w2;

  /*** delete any "n"s in the start of W **/
  while((fout != NULL)&&(fout->el == n))
    {
      w2 = fout->next; 
      free(fout); 
      fout = w2;
    }
  /** list had 0 or only ns so return null ***/
  if(fout == NULL)return(NULL);

  /* fout-> el ! = n */
  prev = fout;
	
  for(w1=fout;w1!= NULL; w1= w1->next)
    {
      if(w1->el == n)
	{
	  prev->next = w1->next;
	  free(w1);
	}
      prev = w1;
    }
  return(fout);
}

/*******************************
deletes  first occurances of "n" from the 
fanout/fanin list "fout";
 it saves a lot of CPU time over Sdelete_W
******************************/

W *delete_W(W *fout,int n)
{
  W *w1, *prev;

  if(fout == NULL)return(NULL);


  prev = fout;
  if(fout->el == n){fout = fout->next;free(prev);return(fout);}

  for(w1=fout;w1!= NULL; w1= w1->next)
    {
      if(w1->el == n)
	{
	  prev->next = w1->next;
	  free(w1);
	  break;
	}
      prev = w1;
    }
  return(fout);
}

/*******************************
delete  first occurances of "n" from the 
linked list for integer "fout"
******************************/

Q *delete_Q(Q *fout,int n)
{
  Q *q1, *prev;

  if(fout == NULL)return(NULL);


  prev = fout;
  if(fout->el == n){fout = fout->next;free(prev);return(fout);}

  for(q1=fout;q1!= NULL; q1= q1->next)
    {
      if(q1->el == n)
	{
	  prev->next = q1->next;
	  free(q1);
	  break;
	}
      prev = q1;
    }
  return(fout);
}

/****************************************
free_Q

free the memory of a linked list
**************************************/

void free_Q(Q *list)
{
  Q *q1, *q2;
  q1  = list;

  while(q1 != NULL)
    {
      q2 = q1->next;
      free(q1);
      q1 = q2;
    }
}
/****************************************
free_W

free the memory of a fanout/fanin list
**************************************/

void free_W(W *list)
{
  W *w1, *w2;
  w1  = list;

  while(w1 != NULL)
    {
      w2 = w1->next;
      free(w1);
      w1 = w2;
    }
}

/****************************************
free_F

free the memory of a "F" (real nnumber) list
**************************************/

void free_F(F *list)
{
  F *f1, *f2;
  f1  = list;

  while(f1 != NULL)
    {
      f2 = f1->next;
      free(f1);
      f1 = f2;
    }
}

/*#####################################
GRAPH ROUTINES
#######################################*/

/**************************************
   free graph

free the Phase A constraint graph (C-graph)
***************************************/
void free_graph()
{
  int i;
  EDGE *ept, *ept1;
  for(i=0; i<noverts;i++)
    {
      ept = verts[i].edges[FORWARD];
      while(ept != NULL)
	{
	  ept1 = ept;
	  ept = ept->next;
	  free(ept1);
	}
      ept = verts[i].edges[REVERSE];
      while(ept != NULL)
	{
	  ept1 = ept;
	  ept = ept->next;
	  free(ept1);
	}
		
    }
  free(verts);
  noverts =0;
  noedges = 0;
}

/****************************** 
get min annd max skew over all latches
and return it in global variables
*******************************/
void min_max_skew()
{
  NODETYPE *np;
  Q *q2;

  PA_max_skew = -INF;
  PA_min_skew = INF;


  for(q2= latches;q2 != NULL;q2= q2->next)
    {

      np = nodes[q2->el];
      if(np->skew <PA_min_skew)PA_min_skew = np->skew;
      if(np->skew > PA_max_skew)PA_max_skew = np->skew;
    }
  if(PA_max_skew <0)PA_diff_skew = -1*PA_min_skew;
  else if(PA_min_skew >0)PA_diff_skew = PA_max_skew;
  else PA_diff_skew = PA_max_skew - PA_min_skew;
}



/****************************************
reset_ckt
rset different flags, variables, delays etc
****************************************/

void reset_ckt()
{
  int i;
  NODETYPE *np;

  for(i=0;i<maxid;i++)if(validnode[i])
    {
      np = nodes[i];
      np->nai = 0;
      np->nao = 0;
      np->level = -INF;
      np->bounds[UPPER] = 0;
      np->bounds[LOWER] = 0;
      np->IsFixed = 0;

      np->maxti = -1*INF;
      np->maxto = -1*INF;
      np->minti = INF;
      np->minto = INF;
      np->rev_maxto = -1*INF;
      np->rev_maxti = -1*INF;
    }
}


/**********************************************
free_all_latches

just removes latches without updating
the fanin/fanout, while converting to C-graph.
May not be in use 
***************************************************/

void free_all_latches()
{
  NODETYPE *np1;
  Q *q1;
  int n1;

  /*fprintf(fperr," freeing all latches ");	*/
  fflush(fperr);

  for(q1= latches;q1 != NULL;q1= q1->next)
    {
      n1 = q1->el;
      if(validnode[n1])
	{
	  np1 = nodes[n1];
	  free(np1->fin);
	  free_W(np1->fout);
	  free(np1);

	  validnode[n1] = 0;
	  nonodes--;
	  if(n1 == maxid-1){
	    maxid--;
	    nonodes--;
	  }
	  else freenodes[++nofreenodes] = n1;

	}
    }
  free_Q(latches);
  latches = NULL;
  nolatches = 0;
  /*fprintf(fperr," DONE \n");
    fflush(fperr);*/
}

/*#####################################
CHECKING ROUTINES
#######################################*/

/************************************
  check_nodeid

just a simple check on the node ids
***************************************/
void check_nodeid()
{
  int i,n,no=0;

  no=0;
  n = 0;
  for(i=0;i<MAX_NODES;i++)if(validnode[i]){
    no++;
    if(i >n)n = i;
  }
  if(no != nonodes){
    printf(" ERR check_nodeid nonodes %d while nodes with validnodes = 1 %d\n",nonodes,no);exit(-1);
  }
  if(n != maxid-1){
    printf(" ERR check_nodeid maxid %d but %d havs validnode %d\n",maxid,n,validnode[n]);exit(-1);
  }
	

  if(0)printf(" Maxid %d max valid id %d nonodes %d no valid nodes %d \n",maxid,n,nonodes,no);

}

/************************************
  check_levels
***************************************/
void check_levels()
{
  NODETYPE *np;
  int i,n,no=0,max=0;

  no=0;
  n = 0;
  for(i=0;i<MAX_NODES;i++)if(validnode[i])
    {
      np = nodes[i];
      if(np->level < 0){printf(" ERR check_level node [%d]%s has level %d\n",i,np->name,np->level);
      no++;
      }
      else n++;
      if(max < np->level)max = np->level;
      /*fprintf(fperr,"L %s %d \n",np->name,np->level);*/
    }
  printf("CHECK_LEVELS ALL Levels  checked %d -tive %d pos max %d\n",no,n,max);
}

/************************************
  check_connection

check  gates (not FFs) for proper fanin & fanouts
***************************************/
void check_connection()
{
  W *w1, *w2;
  NODETYPE *np, *np1;
  int i,done,n,n1;
  int *count_fin;
  int *count_fout;

  count_fin = (int *)CALLOC((nogates+5),sizeof(int));
  count_fout = (int *)CALLOC((nogates+5),sizeof(int));

  for(i=0;i<nogates;i++)
    {
      n = gates[i];
      if(validnode[n])
	{count_fin[i] = nodes[n]->nofin;
	count_fout[i] = nodes[n]->nofout;
	}
      else {count_fin[i] = 0;
      count_fout[i] = 0;
      }
    }

  for(i=0;i<nogates;i++)
    {
      n = gates[i];
      if(validnode[n])
	{
	  np = nodes[n];
	  for(w1 = np->fout;w1 != NULL;w1 = w1->next)
	    {
	      n1 = w1->el;
	      np1 = nodes[n1];
	      done = 0;
	      for(w2 = np1->fin;w2 != NULL;w2 = w2->next)
		{
		  if(w2->el == n){
		    done = 1;
		    count_fin[i]--;
		    break;
		  }
		}
	      if(!done){
		printf(" node [%d]%s NOT in fin of [%d]%s\n",n,np->name,n1,np1->name); 
	      }
	    }

	  for(w1 = np->fin;w1 != NULL;w1 = w1->next)
	    {
	      n1 = w1->el;
	      np1 = nodes[n1];
	      done = 0;
	      for(w2 = np1->fout;w2 != NULL;w2 = w2->next)
		{
		  if(w2->el == n){
		    done = 1;
		    count_fout[i]--;
		    break;
		  }
		}
	      if(!done){
		printf(" node [%d]%s NOT in fout of [%d]%s\n",n,np->name,n1,np1->name); 
	      }
	    }
	}
    }

  for(i=0;i<nogates;i++)
    {
      n = gates[i];
      np = nodes[n];
      if(count_fin[i] != 0){printf("CHECK CONN node [%d]%s has uncaccounted INPUTs %d\n",n,np->name,count_fin[i]); }
      if(count_fout[i] != 0){printf(" node [%d]%s has uncaccounted OUTPUTs %d\n",n,np->name,count_fout[i]);} 
    }
}


/*******************
return a copy of fanin/fanout list "original"
************************/

W * copy_W(W *orignal)
{
  W *w1,*w2,*dummy,*head;


  if(orignal == NULL)return (NULL);/* to insure null copies null */

  head = (W *)MALLOC(1,sizeof(W));
  head->el = orignal->el;
  head->next = NULL;
  head->list = NULL;
  dummy = head;

  for(w1= orignal->next;w1 != NULL;w1= w1->next)
    {
      w2 = (W *)MALLOC(1,sizeof(W));
      w2->el = w1->el;
      w2->wt = w1->wt;
      w2->list = copy_F(w1->list);
      w2->next = NULL;
      dummy->next = w2;
      dummy = w2;
    }
  /* make sure the end of the W is null */
  return(head);
}

/*******************
return a copy of "Q *original"
************************/

Q * copy_Q(Q *orignal)
{
  Q *q1,*q2,*dummy,*head;


  if(orignal == NULL)return (NULL);/* to insure null copies null */

  head = (Q *)MALLOC(1,sizeof(Q));
  head->el = orignal->el;
  head->next = NULL;
  dummy = head;

  for(q1= orignal->next;q1 != NULL;q1= q1->next)
    {
      q2 = (Q *)MALLOC(1,sizeof(Q));
      q2->el = q1->el;
      q2->next = NULL;
      dummy->next = q2;
      dummy = q2;
    }
  /* make sure the end of the Q is null */
  return(head);
}


/*******************
return a copy of "F *original", but with all 
elements in reverse order, (recall F is ordered list);
it is needed to update fanin when FFs move
************************/

F * copy_reverse_F(F *orignal)
{
  F *f1,*f2,*head;


  if(orignal == NULL)return (NULL);/* to insure null copies null */

  head = (F *)MALLOC(1,sizeof(F));
  head->val = orignal->val;
  head->next = NULL;

  for(f1= orignal->next;f1 != NULL;f1= f1->next)
    {
      f2 = (F *)MALLOC(1,sizeof(F));
      f2->val = f1->val;
      f2->next = head;
      head = f2;
    }
  /* make sure the end of the F is null */
  return(head);
}

/*******************
return a copy of "F *original".
************************/

F * copy_F(F *orignal)
{
  F *f1,*f2,*dummy,*head;


  if(orignal == NULL)return (NULL);/* to insure null copies null */

  head = (F *)MALLOC(1,sizeof(F));
  head->val = orignal->val;
  head->next = NULL;
  dummy = head;

  for(f1= orignal->next;f1 != NULL;f1= f1->next)
    {
      f2 = (F *)MALLOC(1,sizeof(F));
      f2->val = f1->val;
      f2->next = NULL;
      dummy->next = f2;
      dummy = f2;
    }
  /* make sure the end of the F is null */
  return(head);
}

/**********************************
remove artificial gates,
Operates on circuit not C-graph, so the circuit
must have explicit latches (not edge weights)
***************************************/

void remove_artificial_gates()
{
  int i,n,in,out,done;
  NODETYPE *np,*inp,*outp;
  W *w1,*w2;


  for(i=0;i<noart;i++)
    {
      n= artificial[i];
      np = nodes[n];
      in = np->fin->el;
      inp = nodes[in];

      /** update fouts */
      for(w1= np->fout; w1 != NULL; w1 = w1->next)
	{
	  out = w1->el;
	  outp = nodes[out];
	  done = 0;
	  for(w2=outp->fin;w2 != NULL; w2= w2->next)
	    {
	      if(w2->el == n)
		{
		  w2->el = in;
		  done = 1;
		  break;
		  /**** !!!! change only one or all */
		}
	    }
	  if(!done) {
	    printf("\n ERR{remove_art} did not find [%d]%s in IN of [%d]%s\n",n,np->name,out,outp->name);
	    exit(0);
	  }
	}

      /* update fin */



      inp->fout = delete_W(inp->fout,n);
      inp->nofout--;
      if(inp->fout == NULL)inp->fout = np->fout;
      else {
	for(w1=inp->fout;w1->next != NULL; w1= w1->next);
	w1->next = np->fout;
      }
      inp->nofout += np->nofout;
      np->fout = NULL;
      np->nofout = 0;
      free(np->fin);
      free(np);
      validnode[n] =0;
    }
  noart =0;
}



/****************************************
set_delays

sets delay for gates, which did not have delay value in the input netlist,
(delay information is optional in the modified MCNC format, and not 
availiable for any gate in the MCNC format)

called after the input netlist is parsed

MODIFY THIS TO CHANGE GATE DELAYs
********************************/
void set_delays(float maxval)
{
  int i,n;
  NODETYPE *np;

  for(i=0;i<nogates;i++)
    {
      n = gates[i];
      np = nodes[n];
      /** only if delay not read in AND if it is a proper gate **/
      if((np->maxdel < 0 )&& (np->type >= And) && (np->type <= Nand)) 
	{
	  if( (np->type >= And) && (np->type <= Nand)) 
	    {
	      np->maxdel = (float) (np->nofout );
	      if(np->maxdel > maxval)np->maxdel = maxval;
	      if(max_gate_delay < np->maxdel)max_gate_delay = np->maxdel;
	    }
	  else np->maxdel = 0;
	}
    }

}

/****************************************
free_ckt()

********************************/

void free_ckt()
{
  int i;
	

  for(i=0;i<maxid;i++)
    {
      if(validnode[i])
	{
	  free_W(nodes[i]->fin);
	  free_W(nodes[i]->fout);
	  free(nodes[i]);

	  validnode[i] = 0;
		
	}
    }
  free_Q(latches);
  latches = NULL;
  nolatches = 0;
  nogates =0;
  nopin =0;
  nopout=0;

}
/******************************
NOT in use 
*******************************/
int count_latch_fout()
{
  int n;
  NODETYPE *np;
  Q *q2;

  int count =0;

  for(q2= latches;q2 != NULL;q2= q2->next)
    {

      n = q2->el;
      if(validnode[n])
	{
	  np = nodes[n];
	  count += np->nofout;
	}
    }
		
  return(count);
}	

/***************** EOF **************/
