/*******************************************************************
*  MINARET (for edge-triggered circuits)
*
*  BY
*
*  NARESH MAHESHWARI AND SACHIN S. SAPATNEKAR
*
*  Copyright 1998 Iowa State University Research Foundation, Inc.
*  All Rights Reserved
*
*  Source Code for retiming edge-triggered circuits
*
*  ASTRA:   Refer to paper in DAC 95 by Deokar & Sapatnekar
*                       and IEEE Tran on CAD 10/1998
*  Minaret: Refer to paper in DAC 97 by Maheshwari & Sapatnekar
*                       and IEEE Tran on VLSI 10/1998
*
*  Contact Address: sachin@ece.umn.edu
*
*        Availiable on as is basis, with no support
*******************************************************************/
/******************************
FILE phase2.c

Performs the actual movements of latches in retiming

Modified in oct 96 to do retiming on edges (i.e., without explicit latches)

POSSIBLE OPTIMIZTION
  maintain edge_counter only for nodes that have duplicate edges
so that I do not have to go and reset every counter before every use,
compare time by using time before & after each reset.  
Can maintain skew list only at fanouts rather than both fan and fanout
(as currently done) but it  may not help much if weights are maintained on both edges
******************************/

#include "include.h"

#include<time.h>

/*******************************
GLOBAL VARS
******************************/

extern int maxdel_latch;
extern int db_ret;
extern int db_ret1;
extern int db_ret0;

/*********************************************
retime(float period,int slack_for , int slack_rev, int bound_dir )

do Phase B of ASTRA 
IMPORTANT
needs all latches (atleast with -tve skew) to have single fanouts

done in 2 passes
1 - all with - skews
2 - all with + skews

ASSUMES that slack information available (ie forward/reverse PERT as needed have been done)
   and retime queue has been set up as needed

period:  target clock period
slack_for: flag, if == 1 then use slack while forward moves
slack_rev: flag, if == 1 then use slack while backward moves
bound_dir: where to store bounds (i.e., ASAP/ALAP)
do_period: falg, if == 1 then minperiod so do minimum motion of latches
********************************************/
int retime(float period,int slack_for , int slack_rev ,int bound_dir, int do_period)
{
  W *w1;
  int n,i;
  NODETYPE *np;
  int count_retime=0;


  set_up_retime_Q();

#ifdef DEBUG
  if(db_ret)printf(" INP retiming ");
  if(db_ret1)printq();
#endif


  /****** procss retime queue **/
  while(top !=NULL)
    {
      n = top->el;
      removeq();
#ifdef DEBUG
      if(db_ret)printf(" INP retiming processing gate [%d]%s",n,nodes[n]->name);
      if(db_ret1)printq();/*for(i=0;i<nogates;i++)print_node(gates[i]);*/

#endif
      retime_gate_inputs(n,slack_for,bound_dir,do_period);
      count_retime++;
    }/* while top != NULL, Q not empty */


  /** forward retiming */

#ifdef DEBUG
  if(db_ret)printf(" NOW Retiming latches at outputs \n\n");
#endif
  top = bot = NULL;
  for(i=0;i<nogates;i++)
    {
      np = nodes[gates[i]];
      np->nao =0;
      for(w1=np->fout;w1 != NULL;w1 = w1->next)
	{ 
	  if(w1->wt > 0)np->nao++; 
	}
      if(np->nao == np->nofout)if(np->type != ARTIFICIAL)addq(gates[i]);
    }


#ifdef DEBUG
  if(db_ret)printf(" going for Q retiming +tive \n");
  if(db_ret1)printq();
#endif

  /** backward retiming */

  while(top !=NULL)
    {
      n = top->el;
      removeq();
#ifdef DEBUG
      if(db_ret)printf("OUT retiming processing gate[%d]%s",n,nodes[n]->name);
      if(db_ret1)printq();/*for(i=0;i<nogates;i++)print_node(gates[i]);*/
#endif

      retime_gate_outputs(n, slack_rev,bound_dir,do_period);
      count_retime++;
    }/* while top != NULL, Q not empty */

  return(count_retime);
}

/**********************************
retime_gate_inputs(int n,int use_slack_mode, int bound_dir )

Retimes a gate which has FFs at all its inputs
moves latch if it can or is required to move, and set skews.
Also adds to the Q any gate (inc itself) which has
FFs at all it's inputs (i.e., is ready to be retimed now)

Method
In forward motion of FFs ( in to out) max skew at inpupts is effective
so get min and max skews at all inputs.
if min > 0 => all + skew; no point in moving forward
              leave them there itself so that they can move back
if max < 0 => all -tive; so can move (but move only if magnitude
              decreases)
 else      => both +s & -s; set all negative skews to 0
              since they can't move forward (because at least on FF has +s)
	      or reverse (as they have -tve skew);
	      the +s can then move backward
***********************************/

void retime_gate_inputs(int n,int use_slack_mode, int bound_dir,int do_period )
{
  W *w1, *w2;
  int n1;
  NODETYPE *np, *np1;
  float maxskew,minskew,gdelay,newskew,slack,max_maxti;
  F *f1,*f2;
  int edge_count;
  int do_move=0;

  if(n == HOST)return;

  np = nodes[n];
  if(np->type == ARTIFICIAL)return;
  gdelay = np->maxdel;
  maxskew = -INF;
  minskew = INF;

  /** check presence of latches and get min and max skews **/
  for(w1=np->fin;w1 != NULL;w1 = w1->next)
    {
#ifdef CHECK
      if(w1->wt <1){
	printf(" ERR{retime_gate_inp} edge from  [%d]%s to [%d]%s has wt %d\n",w1->el,nodes[w1->el]->name,
	       n,np->name,w1->wt);
	printf(" nai = %d fin %d \n",np->nai,np->nofin);
	print_ckt_graph(); 
	fflush(stdout);
	exit(0);
      }
#endif
      if(maxskew < w1->list->val)maxskew = w1->list->val;
      if(minskew > w1->list->val)minskew = w1->list->val;
      /*if(max_maxti < np1->maxti)max_maxti = np1->maxti;*/
    }

#ifdef DEBUG
  if(db_ret)printf("RETIME INPUTS for gate [%d]%s max skew %g min %g Mode %d\n",n,nodes[n]->name,maxskew,minskew,use_slack_mode);
#endif

  /* for FFs at inputs minimum is effective so skew can only be increased */
  if(minskew <0) /* if all skew + moving will only increase skew */
    {
      /* maxskew is the effective now */
      if(maxskew >0)maxskew = 0;/*don't want to increase skew */

      /*** CHANGED TEMP from -gdelay/2 to -gdely; and < to <= 
	changing for minperiod & minarea **/
      if(do_period)
	{if(maxskew <(-1*gdelay/2))do_move = 1; else do_move = 0;}
      else 
	{if(maxskew <= (-1*gdelay))do_move = 1; else do_move = 0;}


      if(do_move&&(np->type != ARTIFICIAL)) /* move latch to fanout unless it is a Pout */
	{
	  np->bounds[bound_dir]--;

	  for(w1=np->fout;w1 != NULL;w1 = w1->next)
	    {
	      nodes[w1->el]->level = 0;
	    }

	  for(w1=np->fout;w1 != NULL;w1 = w1->next)
	    {
	      n1 = w1->el;
	      np1 = nodes[n1];
	      newskew = maxskew + gdelay;

	      if(use_slack_mode)/* USE slack */
		{
		  if(w1->wt == 0)slack = np->rev_maxti - np1->rev_maxto;
		  else slack = np->rev_maxti + w1->list->val;
					
		  if(slack <0){
		    printf("ERROR RET INP NEG SLACK  between [%d]%s to [%d]%s slack %g RI %g RO %g [wt %d skew %g ]\n\n",n,np->name,n1,nodes[n1]->name,slack,np->rev_maxti,np1->rev_maxto,w1->wt,w1->list->val);
		    exit(0);
		  }
		  if(newskew <0)
		    {
		      newskew = newskew + slack;
		      if(newskew >0)newskew = 0;
		    }

		}

	      w1->wt++;
	      if(w1->wt == 1){
		np1->nai++; 
		if(np1->nai == np1->nofin)addq(n1);
	      } /* add only a latch was not already there on this edge */
	      f1 = (F *)MALLOC(1,sizeof(F));
	      f1->val = newskew;
	      f1->next = w1->list;
	      w1->list = f1;

	      edge_count = np1->level;
	      for(w2= np1->fin;w2 != NULL;w2 = w2->next)
		{
		  if(w2->el == n)
		    {
		      if(edge_count == 0)
			{
#ifdef CHECK
			  if(w2->wt <0){
			    printf(" ERR edge wt %d in fanin of [%d]%s to [%d]%s \n",w2->wt,n1,np1->name,n,np->name);
			    print_ckt_graph();
			    exit(0);
			  }
#endif
			  w2->wt++;
			  if(w2->wt == 1)
			    {
			      w2->list = (F *)MALLOC(1,sizeof(F));
			      w2->list->val = newskew;
			      w2->list->next = NULL;
			    }
			  else 
			    {
			      for(f1 = w2->list;f1->next != NULL;f1= f1->next);
			      f2 = (F *)MALLOC(1,sizeof(F));
			      f2->next = NULL;
			      f2->val = newskew;
			      f1->next = f2;
			    }
			  np1->level++;
			  break;
			}
		      else edge_count--;
		    }
		}
	    }

	  if(use_slack_mode)  /* update slack only if was used, so that other slack is not overwritten; don't move this code up as it is used in slack */
	    {
	      np->rev_maxto = -1*maxskew;
	      np->rev_maxti = -1*(maxskew +gdelay) ;
	    }
	  /** update fanin of self and fanouts of fanins */
	  np->nai = 0;
	  for(w1=np->fin;w1 != NULL;w1 = w1->next)
	    {
	      nodes[w1->el]->level = 0; /* for duplicate nodes **/
	      w1->wt--;
	      f1 = w1->list;
	      w1->list = f1->next;
	      free(f1);
	      if(w1->wt >0)np->nai++;
	    }

	  if(np->nai == np->nofin)addq(n);
			
	  for(w1=np->fin;w1 != NULL;w1 = w1->next)
	    {
	      n1 = w1->el;
	      np1 = nodes[n1];
	      edge_count = np1->level;
	      for(w2= np1->fout;w2 != NULL;w2 = w2->next)
		{
		  if(w2->el == n)
		    {
		      if(edge_count == 0)
			{
#ifdef CHECK
			  if(w2->wt <1){
			    printf(" ERR edge wt %d in fanout of [%d]%s to [%d]%s \n",w2->wt,n1,np1->name,n,np->name);
			    print_ckt_graph();
			    exit(0);
			  }
#endif
			  w2->wt--;
			  if(w2->wt == 0){
			    free(w2->list);
			    w2->list = NULL;
			  }
			  else 
			    {
			      for(f1 = w2->list;f1->next->next != NULL;f1= f1->next);
			      free(f1->next);
			      f1->next = NULL;
			    }
			  np1->level++;
			  break;
			}
		      else edge_count--;
		    }
		}
	    }
	}
      else /* no advantage in moving FF */
	{

	  /* do not move latch, but set all fanin skew to maxskew */
	  /* set skew of FFs with -tive skew to be maxskew */

	  for(w1=np->fin;w1 != NULL;w1 = w1->next)
	    { 
	      nodes[w1->el]->level = 0;
	    }


	  for(w1=np->fin;w1 != NULL;w1 = w1->next)
	    {
	      n1 = w1->el;
	      np1 = nodes[n1];
	      /** only neg skews to be updated **/
	      if(w1->list->val <0)w1->list->val = maxskew;
	      edge_count = np1->level;
	      for(w2= np1->fout;w2 != NULL;w2 = w2->next)
		{
		  if(w2->el == n)
		    {
		      if(edge_count == 0)
			{
			  for(f1 = w2->list;f1->next != NULL;f1= f1->next);
			  if(f1->val <0)f1->val = maxskew;
			  np1->level++;
			  break;
			}
		      else edge_count--;
		    }
		}
	    }
	}
    }/* if min <0 */

}


/**********************************
retime_gate_outputs(int n,int use_slack_mode,int bound_dir)

retimes a gate which has FFs at all its outputs;
moves if it can or is required and set skews ;
also adds to the Q any gate (inc itself) which has
FFs at all it's outputs;

In reverse motion of FFs (out to in) min skew at inputs 
is effective
get min and max skews at all inputs
if max < 0 => all - skew; no point in moving reverse
              leave them so that they can move forward
	      
if min > 0 => all + skew; so can move (but move only if magnitude
              decreases)
 else      => both + skew & -s; set all positve skew to 0
              since they can't move reverse (because at least on FF has -tve skew)
	      or forward (as they have +tve skew);
	      the + skew can then move in reverse

option parameters same as retime_gate_inputs.
***********************************/
void retime_gate_outputs(int n,int use_slack_mode, int bound_dir,int do_period)
{
  W *w1, *w2;
  int i,n1,t;
  NODETYPE *np, *np1;
  float maxskew,minskew,gdelay,newskew,slack,max_maxti;
  F *f1,*f2;
  int edge_count;
  int do_move;

  if(n == HOST)return;

  np = nodes[n];
  if(np->type == ARTIFICIAL)return;

  gdelay = np->maxdel;
  maxskew = -INF;
  minskew = INF;
  for(w1=np->fout;w1 != NULL;w1 = w1->next)
    {
#ifdef CHECK
      if(w1->wt <1){
	printf(" ERR{retime_gate_inp} edge from  [%d]%s to [%d]%s has wt %d\n",w1->el,nodes[w1->el]->name,
	       n,np->name,w1->wt);
	printf(" nai = %d fin %d \n",np->nai,np->nofin);
	print_ckt_graph();
	fflush(stdout);
	exit(0);
      }
#endif
      if(maxskew < w1->list->val)maxskew = w1->list->val;
      if(minskew > w1->list->val)minskew = w1->list->val;
      /*if(max_maxti < np1->maxti)max_maxti = np1->maxti;*/
    }

#ifdef DEBUG
  if(db_ret)printf("RETIME OUTPUTS for gate [%d]%s max skew %g min %g slack %d\n",n,nodes[n]->name,maxskew,minskew,use_slack_mode);
#endif

  /* for FFs at inputs max is effective so can only increase the skew */
  if(maxskew >0) /* if all skew + moving will only increase skew */
    {
      /* minskew is effective now */
      if(minskew < 0)minskew = 0; /*don't want to decrease skew */

      if(do_period)
	{if(minskew >gdelay/2)do_move = 1; else do_move = 0;}
      else 
	{if(minskew >= (gdelay))do_move = 1; else do_move = 0;}
      if(do_move&&(np->type != ARTIFICIAL))/* move latch to fanins unless it is a Pin */
	{
	  np->bounds[bound_dir]++;

	  for(w1=np->fin;w1 != NULL;w1 = w1->next)
	    {
	      nodes[w1->el]->level = 0;
	    }

	  for(w1=np->fin;w1 != NULL;w1 = w1->next)
	    {
	      n1 = w1->el;
	      np1 = nodes[n1];
	      newskew = minskew - gdelay;

	      if(use_slack_mode)/* USE slack */
		{
		  if(w1->wt == 0)slack = np->maxti - np1->maxto;
		  else slack = np->maxti - w1->list->val;

		  if(slack <0){
		    printf("ERROR RET INP NEG SLACK  between [%d]%s to [%d]%s slack %g RI %g RO %g [wt %d skew %g ]\n\n",n,np->name,n1,nodes[n1]->name,slack,np->rev_maxti,np1->rev_maxto,w1->wt,w1->list->val);
		    print_ckt_graph(0);
		    exit(0);
		  }
		  if(newskew >0)
		    {
		      newskew = newskew - slack;
		      if(newskew <0)newskew = 0;
		    }

		}

	      w1->wt++;
	      if(w1->wt == 1){
		np1->nao++; 
		if(np1->nao == np1->nofout)addq(n1);
	      } /* add only a latch if one was not already there on this edge */
	      f1 = (F *)MALLOC(1,sizeof(F));
	      f1->val = newskew;
	      f1->next = w1->list;
	      w1->list = f1;

	      edge_count = np1->level;
	      for(w2= np1->fout;w2 != NULL;w2 = w2->next)
		{
		  if(w2->el == n)
		    {
		      if(edge_count == 0)
			{
#ifdef CHECK
			  if(w2->wt <0){
			    printf(" ERR edge wt %d in fanout of [%d]%s to [%d]%s \n",w2->wt,n1,np1->name,n,np->name);
			    print_ckt_graph();
			    exit(0);
			  }
#endif
			  w2->wt++;
			  if(w2->wt == 1)
			    {
			      w2->list = (F *)MALLOC(1,sizeof(F));
			      w2->list->val = newskew;
			      w2->list->next = NULL;
			    }
			  else 
			    {
			      for(f1 = w2->list;f1->next != NULL;f1= f1->next);
			      f2 = (F *)MALLOC(1,sizeof(F));
			      f2->next = NULL;
			      f2->val = newskew;
			      f1->next = f2;
			    }
			  np1->level++;
			  break;
			} /* edge_count*/
		      else edge_count--;
		    }/* w2->el == n */
		}/* all fouts */
	    }

	  if(use_slack_mode)  /* update slack only used, so that other slack is not overwritten*/
	    {
	      np->maxto = minskew; /* don't move up as it is used in slack */
	      np->maxti = minskew - gdelay ;
	    }
	  /** update fanout of self and fanouts of fanins */
	  np->nao = 0;
	  for(w1=np->fout;w1 != NULL;w1 = w1->next)
	    {
	      nodes[w1->el]->level = 0; /* for duplicate nodes **/
	      w1->wt--;
	      f1 = w1->list;
	      w1->list = f1->next;
	      free(f1);
	      if(w1->wt >0)np->nao++;
	    }

	  if(np->nao == np->nofout)addq(n);

	  for(w1=np->fout;w1 != NULL;w1 = w1->next)
	    {
	      n1 = w1->el;
	      np1 = nodes[n1];
	      edge_count = np1->level;
	      for(w2= np1->fin;w2 != NULL;w2 = w2->next)
		{
		  if(w2->el == n)
		    {
		      if(edge_count == 0)
			{
#ifdef CHECK
			  if(w2->wt <1){
			    printf(" ERR edge wt %d in Fanin of [%d]%s to [%d]%s \n",w2->wt,n1,np1->name,n,np->name);
			    print_ckt_graph();
			    exit(0);
			  }
#endif
			  w2->wt--;
			  if(w2->wt == 0){
			    free(w2->list);
			    w2->list = NULL;
			  }
			  else 
			    {
			      for(f1 = w2->list;f1->next->next != NULL;f1= f1->next);
			      free(f1->next);
			      f1->next = NULL;
			    }
			  np1->level++;
			  break;
			}
		      else edge_count--;
		    }
		} /* all fin */
	    }
	}
      else /* no advantage in moving FF */
	{

	  /* do not move latch but set all fanin skew to maxskew */
	  /* set skew of FFs with +tive skew to be maxskew */

	  for(w1=np->fout;w1 != NULL;w1 = w1->next)
	    { 
	      nodes[w1->el]->level = 0;
	    }


	  for(w1=np->fout;w1 != NULL;w1 = w1->next)
	    {
	      n1 = w1->el;
	      np1 = nodes[n1];
	      if(w1->list->val >0)w1->list->val= maxskew;
	      edge_count = np1->level;
	      for(w2= np1->fin;w2 != NULL;w2 = w2->next)
		{
		  if(w2->el == n)
		    {
		      if(edge_count == 0)
			{
			  for(f1 = w2->list;f1->next != NULL;f1= f1->next);
			  if(f1->val >0)f1->val = maxskew;
			  np1->level++;
			  break;
			}
		      else edge_count--;
		    }
		}
	    }
	}
    }/* if max > 0*/

}

/************* EOF *************/
