/*******************************************************************
*  MINARET (for edge-triggered circuits)
*
*  BY
*
*  NARESH MAHESHWARI AND SACHIN S. SAPATNEKAR
*
*  Copyright 1998 Iowa State University Research Foundation, Inc.
*  All Rights Reserved
*
*  Source Code for retiming edge-triggered circuits
*
*  ASTRA:   Refer to paper in DAC 95 by Deokar & Sapatnekar
*                       and IEEE Tran on CAD 10/1998
*  Minaret: Refer to paper in DAC 97 by Maheshwari & Sapatnekar
*                       and IEEE Tran on VLSI 10/1998
*
*  Contact Address: sachin@ece.umn.edu
*
*        Availiable on as is basis, with no support
*******************************************************************/
/*******************************
FILE: hash.c

Contains the hashing routine for parsing the 
input netlist
******************************/
#include "include.h"
#include <math.h>

/*===============================================================*/
/*       Rick's Hashing Stuctures                                */
/*===============================================================*/
/*  I'm going to use a linear hashing routine with a half twist! */
/*  It'll be just like a linear hashing routine for inserting    */
/*  and searching by name.  However... Now here's the half twist,*/
/*  It will also have an index that will look into the hashed    */
/*  list in a sequential manner by number.  This way the list is */
/*  accessible by pin number after the list has been assembled.  */
/*  I'm guessing that the extra storage requirements will not be */
/*  a problem.  Compared to the gargantuan proportions of the    */
/*  rest of the program this is minuscule!                       */
/*===============================================================*/
/*===============================================================*/

/* This is the half twist!  This is an array of integers that point to 
the pins in the hash table sequentially */


/* This makes a bunch of nodes. Just enough to fit the maximum gates. */

/*******************************
GLOBAL VARS
******************************/

struct hashnode hashidx[(int)(MAX_GATES/PercentSplit/PNperND)];
struct pn *pinidx[MAX_GATES];


int nextsplit=0;       /* points to hashidx for next node to split  */
int hashlevel=0;      /* what level is the linear hash on.         */
int hashtotal=0;     /* Total number of pins in the table         */
int hashnodes=0;    /* Total number of hashnodes in the table    */
int newpin;        /* Flag indicating new pin has been created. */
struct pn *newpn;  /* a pointer to a volatile new pin.          */

/*==============================================================*/
/*==============================================================*/


/*=================================================================*/
/*          Rick's Hashing Routines                                */
/*=================================================================*/
/******************************************************************************/
/*  Initializes the hash table.                                               */
/*  Since the smallest design we are using has over 1000 pins I've chosen to  */
/*  start the hash table with InitNDs(defined above) nodes rather than 2.     */
/******************************************************************************/
void  inithash()
{
  int x;
  for(x = 0 ; x < InitNDs ; x++) {
    newnode(&hashidx[x]);
  }
  newpn = (struct pn *)CALLOC(1,sizeof(struct pn));
  newpn->nx = NULL;
  hashnodes = InitNDs;
  hashtotal = 0;
  nextsplit = 0;
  hashlevel = 0;
  newpin   = 0;
}

/******************************************************************************/
/* Newnode  : Clears up a new hash node and returns a pointer to it.          */
/* disposeof: Clears up an old hash node and returns nothing.                 */
/******************************************************************************/
void   newnode(struct hashnode *n)

{
  n->p = NULL;
  n->pins = 0;
  return;
}

void   disposeof(struct pn *n)
{
  struct pn *a,*b;
  if(n == NULL)return;
  for(a = n, b = a->nx ; b != NULL ; a = b,b = a->nx) {
    free(a);
  }
  return;
}

/******************************************************************************/
/*  Returns the number 'key for' the string that is passed to it.  This simply*/
/*  adds the ascii codes together to come up with a key.  I figured it was    */
/*  fast and sufficient for our uses.                                         */
/******************************************************************************/
int keyfor(char *a)

{
  char *b;
  int i=0,x=0;
  for(i = 100000 , b = a ; *b != NULL && i != 1 ; b++) {
    if( *b == 32) continue;
    if( *b >= 65)
      x += *b * i;
    else {
      x += *b * i;
      i /= 10;
    }
  }
  x /= (i*10);
  return x;
}

/******************************************************************************/
/* Takes a pin with name and number.  Puts it into the hash list and links it*/
/* into the array of pointers                                                 */
/******************************************************************************/
/*    This is called by search hash.  But if you know a pin isn't in the hash */
/* as in the case of pi's and po's you can use this to force the pin into the */
/* hash.  It's safe to do this whether it's in there or not.  It will simply  */
/* have a duplicate pin in the node.                                          */
/*   To call this you need to set up a pin structure and pass it a pointer to */
/* it.  It will not distroy the structure of the pin passed to it.            */
/*    inshash(pin);
      /******************************************************************************/
void inshash(struct pn *nd)

{ /* Begin inshash */
  int hky=0;              /*hky=hash index node it fits into.*/
  struct pn *ptr,*ipin;  /* Pointer into the nodes to the individual pins */
  nd->nx = NULL;
  /***** get node to put it in */
  hky = nd->key % (InitNDs*((int)pow((double)2,(double)hashlevel)));

  /***** I don't want to go into explaining linear hashing */
  if (hky < nextsplit)
    hky = nd->key % (InitNDs*((int)pow((double)2,(double)(hashlevel+1))));

  /**** copy the new pin into the hash node */

  hashidx[hky].pins++;

  /* this will look for the end of the node and make a new pin there. */
  if(hashidx[hky].pins == 1) {
    hashidx[hky].p = (struct pn *)CALLOC(1,sizeof(struct pn));
    ipin = hashidx[hky].p;
    hashidx[hky].p->nx = NULL;
  }
  else {
    for(ptr = hashidx[hky].p; ptr->nx != NULL ; ptr = ptr->nx);
    ptr->nx = (struct pn *)CALLOC(1,sizeof(struct pn));
    ipin = ptr->nx;
    ptr->nx->nx = NULL;
  }

  /* This copies the information into the new pin */
  *ipin = *nd;
  ipin->nx = NULL;
  strcpy(ipin->name,nd->name);

  /**** Here's the half twist again !  This makes a pointer
    to the pin inside the node from an array of ptrs*/
  pinidx[nd->num] = ipin;

  /**** Keeps track of total pins in the hash table. */
  hashtotal++;


  /***** Check storage for split  80% Max *****/
  /***** Taking into consideration clustering and so forth
    This will give at least 70% usage.  I hope this
    Will not give any really long search chains.
    I'll check of course! */
  if(hashtotal/(hashnodes*PNperND) > PercentSplit)
    splithashnode();
  return;
}

/******************************************************************************/
/* This splits nodes for linear hashing.  Like I said before I don't want to  */
/* go into explaining linear hashing.                                         */
/******************************************************************************/
void   splithashnode()
{
  int xx,vv,savetotal=0;
  struct pn *ptr,*tpt,*tpn;  /* A bunch of temporary pin pointers */

  /***** Keeps track of total nodes */
  hashnodes++;
  /***** This disallows splitting while I'm rehashing
    The pins in this node. Also doesn't add the
    rehashed nodes twice to the total */
  savetotal = hashtotal;
  hashtotal = 0;
  xx = nextsplit;

  /***** Move next split pointer *****/
  /***** If nextsplit is at the end of the level then
    Reset nextsplit and allocate next level of nodes */
  if(nextsplit == (InitNDs*(int)pow((double)2,(double)(hashlevel))-1)) {
    hashlevel++;
    nextsplit = 0;
  } else
    nextsplit++;
  /***** Pull pins out of the node to be rehashed.  */

  tpt = hashidx[xx].p;

  /**** Clear up old and new hash nodes */
  newnode(&hashidx[xx]);
  newnode(&hashidx[InitNDs*(int)pow((double)2,(double)(hashlevel))+nextsplit-1]);

  /*****  Rehash all pins in the separated node */
  for(tpn = tpt ; tpt != NULL; tpn = tpt) {
    tpt = tpn->nx;
    inshash(tpn);
    free(tpn);
  }
    
  /*****  Restore total pins */
  hashtotal = savetotal;

  return;
}  /* End if split */
/******************************************************************************/
/*  Search hash will look into the table and find a matching name.  If it does*/
/*  not find the name it will create a new pin and insert it into the table. */
/*  Then it will return the number that it created.  Otherwise it will return */
/*  The number that it finds in the table.                                    */
/******************************************************************************/
int searchhash(char *look,int PiPo)
{

  /* Begin Search */
  int zz=0,key = 0,hky = 0,foundnum=-1;
  struct pn *ptr;
  int modno;



  /* If this is the first time through initialize the hash. */
  if(hashnodes == 0)
    inithash();

  /* Calculate key for the name asked for and figure node # */
  key = keyfor(look);
  modno =  (InitNDs*(int)(pow((double)2,(double)hashlevel)));

  hky = key % modno;

  /*printf(" key %d hky %d hl %d mod %d \n",key,hky,hashlevel,modno);
   */
  if (hky < nextsplit)
    hky = key % (InitNDs*((int)pow((double)2,(double)(hashlevel+1))));

  /***** Search for name through end of pins in the node */
  for(ptr = hashidx[hky].p ; ptr != NULL ; ptr = ptr->nx)
    if(ptr->key==key && !strcmp(ptr->name,look)) {
      foundnum = ptr->num;
      newpin = 0;
      break;
    } /* End if */
  /***** If not found create it!  */
  if( foundnum == -1 ) {

    /***** Set new pin flag */
    newpin = 1;
    /***** Create new pin and put it into the table */
    strcpy(newpn->name,look);
    newpn->key = key;
    /***** Set return value and new pin number.*/
    foundnum = hashtotal;
    
    newpn->num = foundnum;

    /**** Set the PiPo flag on the pin */
    newpn->IsPiPo = PiPo;

    /**** Insert it into the hash */
    inshash(newpn);

    init_node(foundnum,look);

  } /* end create if */

  return foundnum;
} /* End search */

/*=================================================================*/
/*=================================================================*/

/************************************
Added by naresh
just allocates memory for a node "k" (
a gate or a latch or a PIO) and 
initializes some information
****************/

int init_node(int k, char *name)

{
  int i;
  NODETYPE *np;
  /* allocate mem for node */
  np =(NODETYPE *)CALLOC(1,sizeof(NODETYPE));

  if(np== NULL){printf(" NO MEM \n");exit(0);}
  /* initilize node */
  np->type = 0;
  np->nofin = 0;
  np->fin = NULL;
  np->nofout = 0;
  np->fout = NULL;
  np->skew = 0;
  np->phase = -1;
  np->level = -INF;
  np->nai = 0;
  np->mindel = 0;
  np->maxdel = 0;
  np->minto =INF;
  np->maxto = -INF;
  np->rev_maxto = -INF;
  np->rev_maxti = -INF;
  np->minti = INF;
  np->maxti = -INF;
  np->bounds[0] =0;
  np->bounds[1] =0;
  np->IsFixed =0;
  np->id = k;
  strcpy(np->name,name);
  nodes[k] = np;
  validnode[k] = 1;
  nonodes++;

}

void destroy_hash()
{
  int i;
  for(i=0;i<hashtotal; i++)free(pinidx[i]);
  free(newpn);

}
