/*******************************************************************
*  MINARET (for edge-triggered circuits)
*
*  BY
*
*  NARESH MAHESHWARI AND SACHIN S. SAPATNEKAR
*
*  Copyright 1998 Iowa State University Research Foundation, Inc.
*  All Rights Reserved
*
*  Source Code for retiming edge-triggered circuits
*
*  ASTRA:   Refer to paper in DAC 95 by Deokar & Sapatnekar
*                       and IEEE Tran on CAD 10/1998
*  Minaret: Refer to paper in DAC 97 by Maheshwari & Sapatnekar
*                       and IEEE Tran on VLSI 10/1998
*
*  Contact Address: sachin@ece.umn.edu
*
*        Availiable on as is basis, with no support
*******************************************************************/
/*******************************
FILE graph.c
routines for making the Phase A S-graph
******************************/

#include "include.h"


/*******************************
GLOBAL VARS
******************************/

/* global as used by both add_edge & complete_graph
   these store edges to/from PIO
   in seperate data structure; complete_graph
   converts them back to normal; this is done to avoid
   multiple updates on the PIO node.
   */

typedef struct pion
{  
	float todelay;
	float fromdelay;
} PIONODE;

PIONODE *pionode;	/* for pio node; later use dynamic
				   allocation in init_graph and 
				   free in complete graph
				used as an array of size nonodes */


extern int db_graph;


/*****************************
init_graph

just initilize (allocate memory etc)
!!! IMPORTANT pin_name from yacc.y used for host vertex
ONLY one vertex for PIO (not two)
***********************************/
void init_graph()
{
	int i;
	noverts = 0; /* init vert 0 is pio*/
	noedges = 0;

	verts = (VERT *)MALLOC((nolatches+2),sizeof(VERT));
	
	verts[noverts].nodeid = pin_name;	/* to get correct name by nodes[id] */
	verts[noverts].skew = 0;
	verts[noverts].edges[FORWARD] = NULL;
	verts[noverts].edges[REVERSE] = NULL;
	noverts++;

	for(i=0;i<maxid;i++)/* look at all nodes */
	{
		if(validnode[i])
		{
			if(nodes[i]->type == Pin)	/* Pin is vert 0 */
			{
				node2vert[i] = 0;
			}
			else if(nodes[i]->type == Pout)	/* Pout is vert 0 */
			{
				node2vert[i] = 0;
			}
			else if(nodes[i]->type == Latch)
			{
				verts[noverts].nodeid = i;
				verts[noverts].skew = 0;
				node2vert[i] = noverts;
				verts[noverts].edges[FORWARD] = NULL;
				verts[noverts].edges[REVERSE] = NULL;
				noverts++;
			}
			else node2vert[i]= -1; /* for gates */
		}
	}

	pionode = (PIONODE *)MALLOC((nolatches+2),sizeof(PIONODE));
	for(i=0; i<nolatches+2;i++)
	{ 
		pionode[i].todelay = -1;
		pionode[i].fromdelay = -1;
	}

#ifdef DEBUG
	if(db_graph)for(i=0;i<maxid;i++)/* look at all nodes */
	{
		if(validnode[i])printf(" node %d vert %d\n",i,node2vert[i]);
		else printf("invalid node vert %d\n",node2vert[i]);
	}
#endif

}


/********************************************
add_edge(int from,int to,float delay)

   adds edge from "from" to "to" of delay "delay"
   checks for either (to/from) being PIO and uses
   pionode for that
   ***********************************/

void add_edge(int from,int to,float delay)
{
	int vt,vf;	/* vertex no of from & to nodes */
	EDGE *ept;


	/* check if both nodes are latches */
	if(!((nodes[from]->type == Latch)||(nodes[from]->type == Pin)))
	{
		printf(" ERR node %s type %d given to add edge from\n",
		    nodes[from]->name,nodes[from]->type); 
		exit(0);
	}
	if(!((nodes[to]->type == Latch)||(nodes[to]->type == Pout)))
	{
		printf(" ERR node %s type %d given to add edge TO\n",
		    nodes[to]->name,nodes[to]->type); 
		exit(0);
	}

	/* get vertex no; pio's no is 0 */
	vt = node2vert[to];
	vf = node2vert[from];
	if(db_graph)printf(" vt %d vf %d delay %g\n",vt,vf,delay);
	if(vt ==vf) if(Max_self_loop <delay)Max_self_loop = delay;

	if(vf ==0)	/* if vf ==0 ie from is pio so use pionode */
	{
		if(pionode[vt].todelay < delay)pionode[vt].todelay = delay;
	}
	/* if vt ==1 ie to is pio so use pionode 
		    if vt=1;vf =0 todelay is used so pio to pio
		    edges are in todelay*/
	else if(vt==0)
	{
		if(pionode[vf].fromdelay < delay)pionode[vf].fromdelay = delay;
	}

	else /* neither from/to is pio  so add edge */
	{
		ept = (EDGE *)malloc(1*sizeof(EDGE));
		ept->delay = delay;
		ept->next = verts[vf].edges[FORWARD];
		verts[vf].edges[FORWARD] = ept;
		ept->tovert = vt;
		noedges++;
	}
}

/*******************************************
complete_graph()

   completes the graph by adding edges "from" and "to" pio
   this information is stored in pionode which is done away now
   ***************************************/
void complete_graph()
{
	EDGE *ept, *first=NULL;
	int i;

#ifdef DEBUG	
	if(db_graph)for(i=0;i<noverts;i++)printf("i %d to %g from %g\n",i,pionode[i].todelay,pionode[i].fromdelay);
#endif
	for(i=0;i<noverts;i++) /* for all verts */
	{
		if(pionode[i].fromdelay >=0) /* delays are only+ so -means no edge */
		{
			ept = (EDGE *)MALLOC(1,sizeof(EDGE));
			ept->tovert = 0;	/* edge is from vert i to 0 */
			ept->delay = pionode[i].fromdelay;
			ept->next = verts[i].edges[FORWARD];
			verts[i].edges[FORWARD] = ept;
			noedges++;
		}

		if(pionode[i].todelay >=0) /* delays are only+ so -means no edge */
		{
			ept = (EDGE *)MALLOC(1,sizeof(EDGE));
			ept->tovert = i;	/* edge is from vert 0 to i */
			ept->delay = pionode[i].todelay;
			ept->next = first;
			first = ept;
			noedges++;
		}

	}
	/* this may overwrite any perivious edges on vert 0 */
	verts[0].edges[FORWARD] = first;

	/* free pionode */
	free(pionode);
}

/*************************************
check_connect_graph(int Dir)

checks the connecitivity of the graph in the given direction
********************************/
void check_connect_graph(int Dir)
{
	EDGE *ep;
	int i,to,n;
	int *visited;
	int *inq;
	int noarc=0;

	visited = (int *)CALLOC(noverts+2,sizeof(int));
	inq = (int *)CALLOC(noverts+2,sizeof(int));

#ifdef SAFE
	for(i=0;i<noverts;i++) /* for all verts */
	{ 
		visited[i] = 0;
		inq[i] =0;
	}
#endif
	if(0)printf("\n VISITING verts %d  EDGES Dir %d\n",noverts,Dir);
	top = bot = NULL;
	addq(0);
	inq[0]=1;
	while(top !=NULL)
	{
		n = top->el;
		removeq();
		visited[n] = 1;
		if(0)printf("\n visited %d nh ",n);
		for(ep= verts[n].edges[Dir];ep!=NULL;ep= ep->next)
		{
			noarc++;
			to = ep->tovert;
			if(0)printf(" %d v%d |",to,visited[to]);
			if(inq[to] == 0){
				addq(to);
				inq[to] = 1;
			}
			/*printq();*/
		}
	}
	printf("\n arcs %d CHECK \n ",noarc);
	for(i=0;i<noverts;i++)if(visited[i] != 1)printf(" vert %d of node [%d]%s not visited its visted is %d\n",i,verts[i].nodeid,nodes[verts[i].nodeid]->name,visited[i]);

free(visited);
free(inq);

}


/***********************************************
 make_reverse_graph()

needed for bound calc in minarea
*****************************************/

void make_reverse_graph()
{
	EDGE *ep, *ep1;
	int i,to;

	if(0)printf("no verts %d no latches %d\n",noverts,nolatches);

	for(i=0;i<noverts;i++) /* for all verts */
	{
		for(ep= verts[i].edges[FORWARD];ep!=NULL;ep= ep->next)
		{
			ep1 = (EDGE *)MALLOC(1,sizeof(EDGE));
			to = ep->tovert;
			ep1->tovert = i;
			ep1->delay = ep->delay;
			ep1->next = verts[to].edges[REVERSE];
			verts[to].edges[REVERSE] = ep1;
		}
	}
}


/***********EOF*****************/
