/*******************************************************************
*  MINARET (for edge-triggered circuits)
*
*  BY
*
*  NARESH MAHESHWARI AND SACHIN S. SAPATNEKAR
*
*  Copyright 1998 Iowa State University Research Foundation, Inc.
*  All Rights Reserved
*
*  Source Code for retiming edge-triggered circuits
*
*  ASTRA:   Refer to paper in DAC 95 by Deokar & Sapatnekar
*                       and IEEE Tran on CAD 10/1998
*  Minaret: Refer to paper in DAC 97 by Maheshwari & Sapatnekar
*                       and IEEE Tran on VLSI 10/1998
*
*  Contact Address: sachin@ece.umn.edu
*
*        Availiable on as is basis, with no support
*******************************************************************/
/*******************************
FILE bound.c

Routine to obtain bounds on the gate lookup maheshwari & Sapatnekar TVLSi 3/98
for details.
As a side effect of bound calculation the feasiablity of the target clock period is
checked, but in Shenoy's method it is not checked untill the LP is solved.
******************************/

#include "include.h"


/*******************************
GLOBAL VARS
******************************/
extern int Latch_Patch[200];
extern int Latch_Patch_Index;
/** to patch
  latches not reachable from PO but from PI by 
  setting there bounds == ASAP stored in bell.c
  already have INF skews set to 0 **/

extern int FAST_BELL;
extern int db_area;
int Latch_Remove_Counter;
int Latch_Add_Counter;
int Min_L= INF, Max_U=-INF;

/**************************************
IMPORTANT assumes single fanout latches;

Restores the location of latches after calculating the
bounds (bounds[LOWER][UPPER]), since finding bounds requires moving 
latches but bounds are with respect to the initial location of latches,
so initial location of latches need to be restored.
However the convertion to C-graph happens in this function
Will use the function in cnvt.c to store latch location and skew.

Both ASAP and ALAP skews (and the slacks) are calculated first and stored.
Phase B is performed for ASAP skews first to obtain the upper bounds 
which are then saved in "bounds[UPER]",
the original latch location are then restored with ALAP skews and phase B 
performed again. Care is taken not to overwrite any skew or slack value.
A simpler way would have been to do both Phase A and B on ASAP first, then 
reset the circuit and do the ALAP, this would, however, require two runs of 
"all_pert" to find the all-pair input-output delays. (I did not try to measure
howmuch advantage the current method gives in CPU time)

FORWARD is lower bound (in to out)
REVERSE is upper bound (out to in)
****************************************/
int get_bounds(float period)
{

  int i,n;
  int temp1;
  NODETYPE *np;
  float bell_return;
	

  all_pert();

  make_reverse_graph();


#ifdef CHECK
  check_connect_graph(FORWARD);
  check_connect_graph(REVERSE);
#endif

  bell_return = bellford(period,-1*INF,FORWARD);
  if(bell_return != 0){
    printf(" bell returned %.2f for period %.2f\n",bell_return,period);
    exit(0);
  }

  normalise_skews(FORWARD,0); /* copy skew for PERT */
  normalise_skews(FORWARD,1); /* store the skews, here they are not overwritten later**/

  bell_return = forwardpert(1);  /*** storing the slack information so retiming should not overwrite these values in retime_a_gate (they (maxti..) are updated if slack info is used) */

  bell_return = bellford(period,-1*INF,REVERSE);
  if(bell_return != 0){
    printf("REV bell returned %.2f for period %.2f\n",bell_return,period);
    exit(0);
  }
  normalise_skews(REVERSE,0);  /* move skew to latches */

  bell_return = reversepert(1);

  /**** converting to C-graph here **/
  ckt2graph();
  remove_pio();
  free_all_latches();

  /*** PATCH for not multiplying latches before Bellman-Ford, update skews on multiplied latches
    for ASAP (reverse + slack) using rev_maxto at fanout node */
  if(FAST_BELL)update_skews_with_slack_for_reverse_retiming(0, period);

#ifdef SAFE
  /***** since reding in ckt should do it and should not be needed in shenoy ***/
  for(i=0;i<nogates;i++) /* so that host is also taken care of */
    {  
      n= gates[i];
      np = nodes[n];
      np->bounds[LOWER] = 0;
      np->bounds[UPPER] = 0;
      np->IsFixed = 0;
    }
	
#endif
  /*** copy saved latch location and skews **/
  restore_ckt_graph(0);

  temp1 = retime(period,1,0,UPPER,0);  /** ASAP ***/
	
  /*** copy saved latch location and skews **/
  restore_ckt_graph(1);

  temp1 = retime(period,0,1,LOWER,0); /*** ALAP **/

  /*** Check for IMMOBILITY **/

  Min_L = INF; 
  Max_U = -INF;
  for(i=0;i<nogates;i++)
    {
      n= gates[i]; 
      np = nodes[n];
      if(np->bounds[LOWER] !=  np->bounds[UPPER])
	{
	  if(Min_L > np->bounds[LOWER])Min_L = np->bounds[LOWER];
	  if(Max_U < np->bounds[UPPER])Max_U = np->bounds[UPPER];
	}
      else { np->IsFixed = 1;}
		
#ifdef CHECK
      if(np->type == ARTIFICIAL)if(np->bounds[LOWER] !=0 ||np->bounds[UPPER] != 0){
	printf(" ERR artificial node [%d]%s has Bounds L %d U %d\n",np->bounds[LOWER],np->bounds[UPPER]);
	exit(0);
      }
#endif
    }

  restore_ckt_graph(-1); /** restore to original location **/

  free_saved_latches();

  return(0);
}


/******* EOF *************/
